# Copyright (c) 2014 Baidu.com, Inc. All Rights Reserved
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions
# and limitations under the License.

"""
This module provides a client class for parameter template.
"""

import copy
import json
import logging
import uuid

from baidubce import bce_base_client
from baidubce.auth import bce_v1_signer
from baidubce.http import bce_http_client
from baidubce.http import handler
from baidubce.http import http_methods

from baidubce.utils import required
from baidubce import compat

_logger = logging.getLogger(__name__)

class TemplateClient(bce_base_client.BceBaseClient):
    """
    Tempalte base sdk client
    """

    prefix = b'/v1'

    def __init__(self, config=None):
        '''
        :param config:
        :type config: baidubce.BceClientConfiguration
        :return:
        '''
        bce_base_client.BceBaseClient.__init__(self, config)

    def _merge_config(self, config=None):
        """
        :param config:
        :type config: baidubce.BceClientConfiguration
        :return:
        """
        if config is None:
            return self.config
        else:
            new_config = copy.copy(self.config)
            new_config.merge_non_none_values(config)
            return new_config

    def _send_request(self, http_method, path,
                      body=None, headers=None, params=None,
                      config=None, body_parser=None):
        '''
        :param http_method:
        :type http_method: string

        :param path:
        :type path: string

        :param body:
        :type body: dict

        :param headers:
        :type headers: dict

        :param params:
        :type params: dict

        :param config:
        :type config: baidubce.BceClientConfiguration

        :param body_parser:
        :type body_parser: function

        :return: baidubce.BceResponse
        '''
        config = self._merge_config(config)
        if body_parser is None:
            body_parser = handler.parse_json
        if headers is None:
            headers = {b'Accept': b'*/*', b'Content-Type': b'application/json;charset=utf-8'}
        return bce_http_client.send_request(
            config, bce_v1_signer.sign, [handler.parse_error, body_parser],
            http_method, TemplateClient.prefix + path, body, headers, params)

    @required(name=(bytes, str), ip_version=(bytes, str), ip_address_info=list)
    def create_ip_set(self, name, ip_version, ip_address_info, description=None, client_token=None, config=None):
        """
        Create a new ip set.
        :param name:
            The name of ip set to be created.
        :type name: string

        :param ip_version:
            The IP version of the ip set.
        :type ip_version: string

        :param ip_address_info:
            The list of IpAddressInfo to be created in this ip set.
        :type ip_address_info: List[IpAddressInfo]

        :param description:
            The description of the ip set.
        :type description: string

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipSet'
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token

        body = {
            'name': compat.convert_to_string(name),
            'ipVersion': compat.convert_to_string(ip_version),
        }

        if ip_address_info is not None:
            address_info_set = []
            for ip_address in ip_address_info:
                address_info_set.append({"ipAddress": ip_address.ip_address, "description": ip_address.description})
            body['ipAddressInfo'] = address_info_set
        if description is not None:
            body['description'] = compat.convert_to_string(description)

        return self._send_request(http_methods.POST, path, body=json.dumps(body), params=params,
                                  config=config)

    @required(ip_set_id=(bytes, str), ip_address_info=list)
    def add_ip_address_to_ip_set(self, ip_set_id, ip_address_info, client_token=None, config=None):
        """
        Add a list of IpAddressInfo to specified ip set.

        :param ip_set_id:
            The id of the ip set which you want to add IpAddressInfo into.
        :type ip_set_id: string

        :param ip_address_info:
            The list of IpAddressInfo to be added in this ip set.
        :type ip_address_info: List[IpAddressInfo]

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipSet/' + compat.convert_to_string(ip_set_id) + b'/ipAddress'
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token

        body = {
            "ipAddressInfo": []
        }

        if ip_address_info is not None:
            for ip_address in ip_address_info:
                body["ipAddressInfo"].append({"ipAddress": ip_address.ip_address, 
                                              "description": ip_address.description})

        return self._send_request(http_methods.POST, path, body=json.dumps(body), params=params, config=config)

    
    @required(ip_set_id=(bytes, str), ip_address=list)
    def delete_ip_address(self, ip_set_id, ip_address, client_token=None, config=None):
        """
        Delete a list of IpAddressInfo from specified ip set.

        :param ip_set_id:
            The id of the ip set which you want to delete IpAddressInfo into.
        :type ip_set_id: string

        :param ip_address:
            The list of IpAddress to be deleted in this ip set.
        :type ip_address: List[string]
        
        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipSet/' + compat.convert_to_string(ip_set_id) + b'/deleteIpAddress'
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token
        body = {
            "ipAddressInfo": []
        }

        if ip_address is not None:
            for address in ip_address:
                body["ipAddressInfo"].append(address)

        return self._send_request(http_methods.POST, path, body=json.dumps(body), params=params, config = config)

    @required(ip_set_id=(bytes, str))
    def update_ip_set(self, ip_set_id, name=None, description=None, client_token=None, config=None):
        """
        Update the specified IpSet.

        :param ip_set_id:
            The id of the IpSet which you want to update.
        :type ip_set_id: string

        :param name:
            The new name of this IpSet.
        :type name: string

        :param description:
            The new description of this IpSet.
        :type description: string

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """ 
        path = b'/ipSet/' + compat.convert_to_string(ip_set_id)
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token
        params[b'modifyAttribute'] = None

        body = {
            "name": name,
            "description": description
        }

        return self._send_request(http_methods.PUT, path, body=json.dumps(body), params=params, config=config)

    @required(ip_set_id=(bytes, str))
    def delete_ip_set(self, ip_set_id, client_token=None, config=None):
        """
        Delete the specified IpSet.

        :param ip_set_id:
            The id of the IpSet which you want to update.
        :type ip_set_id: string

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """

        path = b'/ipSet/' + compat.convert_to_string(ip_set_id)
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token

        return self._send_request(http_methods.DELETE, path, params=params, config=config)

    def list_ip_set(self, ip_version=None, marker=None, max_keys=None, config=None):
        """
        Return a list of all IpSets owned by the authenticated user.

        :param ip_version:
            The version of IP addresses that are returned.
            Valid values are IPv4 and IPv6.
        :type ip_version: string

        :param marker:
            The optional parameter marker specified in the original request to specify
            where in the results to begin listing.
            Together with the marker, specifies the list result which listing should begin.
            If the marker is not specified, the list result will listing from the first one.
        :type marker: string

        :param max_keys:
            The optional parameter to specifies the max number of list result to return.
            The default value is 1000.
        :type max_keys: int

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipSet'
        params = {"maxKeys": 1000}

        if ip_version is not None:
            params[b'ip_version'] = ip_version
        if marker is not None:
            params[b'marker'] = marker
        if max_keys is not None:
            params[b'maxKeys'] = max_keys
        return self._send_request(http_methods.GET, path, params=params, config=config)
    
    @required(ip_set_id=(bytes, str))
    def get_ip_set_detail(self, ip_set_id, config=None):
        """
        Get the detail information of specified IpSet.

        :param ip_set_id:
            The id of the IpSet which you want to query.
        :type ip_set_id: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipSet/' + compat.convert_to_string(ip_set_id)
        return self._send_request(http_methods.GET, path, config=config)
        
    @required(name=(bytes, str), ip_version=(bytes, str), ip_set_ids=list)
    def create_ip_group(self, name, ip_version, ip_set_ids, description=None, client_token=None, config=None):
        """
        Create an IpGroup.

        :param name:
            The name of the IpGroup.
        :type name: string

        :param ip_version:
            The version of IP addresses that are returned.
            Valid values are IPv4 and IPv6.
        :type ip_version: string

        :param ip_set_ids:
            A list of IpSet ids.
        :type ip_set_ids: list

        :param description:
            The description of the IpGroup.
        :type description: string

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipGroup'
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token

        body = {
            "name": name,
            "ipVersion": ip_version,
            "ipSetIds": ip_set_ids
        }
        if description is None:
            body["description"] = description

        return self._send_request(http_methods.POST, path, body=json.dumps(body), params=params, config=config)

    @required(ip_group_id=(bytes, str), ip_set_ids=list)
    def bind_ip_set(self, ip_group_id, ip_set_ids, client_token=None, config=None):
        """
        Bind IpSet to IpGroup.

        :param ip_group_id:
            The id of the IpGroup which you want to bind IpSet.
        :type ip_group_id: string

        :param ip_set_ids:
            A list of IpSet ids.
        :type ip_set_ids: list

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipGroup/' + compat.convert_to_string(ip_group_id) + b'/bindIpSet'
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token

        body = {
            "ipSetIds": ip_set_ids
        }

        return self._send_request(http_methods.POST, path, body=json.dumps(body), params=params, config=config)

    @required(ip_group_id=(bytes, str), ip_set_ids=list)
    def unbind_ip_set(self, ip_group_id, ip_set_ids, client_token=None, config=None):
        """
        Unbind IpSet from IpGroup.

        :param ip_group_id:
            The id of the IpGroup which you want to remove IpSet.
        :type ip_group_id: string

        :param ip_set_ids:
            A list of IpSet ids.
        :type ip_set_ids: list

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipGroup/' + compat.convert_to_string(ip_group_id) + b'/unbindIpSet'
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token

        body = {
            "ipSetIds": ip_set_ids
        }

        return self._send_request(http_methods.POST, path, body=json.dumps(body), params=params, config=config)
        
    @required(ip_group_id=(bytes, str))
    def update_ip_group(self, ip_group_id, name=None, description=None, client_token=None, config=None):
        """
        Update the IpGroup.

        :param ip_group_id:
            The id of the IpGroup which you want to update.
        :type ip_group_id: string

        :param name:
            The new name of the IpGroup.
        :type name: string

        :param description:
            The new description of the IpGroup.
        :type description: string

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipGroup/' + compat.convert_to_string(ip_group_id)
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token
        params[b'modifyAttribute'] = None

        body = {}
        if name is not None:
            body['name'] = name
        if description is not None:
            body['description'] = description

        return self._send_request(http_methods.PUT, path, body=json.dumps(body), params=params, config=config)

    @required(ip_group_id=(bytes, str))
    def delete_ip_group(self, ip_group_id, client_token=None, config=None):
        """
        Delete the specified IpGroup.

        :param ip_group_id:
            The id of the IpGroup which you want to update.
        :type ip_group_id: string

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """

        path = b'/ipGroup/' + compat.convert_to_string(ip_group_id)
        params = {}
        if client_token is None:
            params[b'clientToken'] = generate_client_token()
        else:
            params[b'clientToken'] = client_token

        return self._send_request(http_methods.DELETE, path, params=params, config=config)

    def list_ip_group(self, ip_version=None, marker=None, max_keys=None, config=None):
        """
        Return a list of all IpGroups owned by the authenticated user.

        :param ip_version:
            The version of IP addresses that are returned.
            Valid values are IPv4 and IPv6.
        :type ip_version: string

        :param marker:
            The optional parameter marker specified in the original request to specify
            where in the results to begin listing.
            Together with the marker, specifies the list result which listing should begin.
            If the marker is not specified, the list result will listing from the first one.
        :type marker: string

        :param max_keys:
            The optional parameter to specifies the max number of list result to return.
            The default value is 1000.
        :type max_keys: int

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipGroup'
        params = {"maxKeys": 1000}

        if ip_version is not None:
            params[b'ip_version'] = ip_version
        if marker is not None:
            params[b'marker'] = marker
        if max_keys is not None:
            params[b'maxKeys'] = max_keys
        return self._send_request(http_methods.GET, path, params=params, config=config)

    @required(ip_group_id=(bytes, str))
    def get_ip_group_detail(self, ip_group_id, config=None):
        """
        Get the detail information of specified IpGroup.

        :param ip_group_id:
            The id of the IpGroup which you want to query.
        :type ip_group_id: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/ipGroup/' + compat.convert_to_string(ip_group_id)
        return self._send_request(http_methods.GET, path, config=config)

def generate_client_token_by_uuid():
    """
    The default method to generate the random string for client_token
    if the optional parameter client_token is not specified by the user.

    :return:
    :rtype string
    """
    return str(uuid.uuid4())


generate_client_token = generate_client_token_by_uuid
