/**
 * Copyright 2014 (c) Baidu, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

#include <sstream>
#include <vector>
#include <json/json.h>

#include "bcesdk/common/common.h"
#include "bcesdk/util/util.h"
#include "bcesdk/bos/model/bucket_meta_data.h"

BEGIN_CPPSDK_NAMESPACE

int BucketDirPaths::serialize_to_json(
       const BucketDirPaths &st, std::string &json_str) {
    try {
        Json::Value root;
        for (const auto &each : st.dir_paths) {
            Json::Value jdir_quota = Json::Value(each).asString();
            root["dirPaths"].append(jdir_quota);
        }
        Json::FastWriter writer;
        json_str = writer.write(root);
        LOGF(WARN, "dir_paths json str:%s", json_str.c_str());
    } catch (...) {
        LOGF(ERROR, "serialize BucketDirPaths to json exceptional!");
        return RET_ILLEGAL_OPERATION;
    }
    return RET_OK;
}

int BucketQuotaInfo::serialize_to_json(
       const BucketQuotaInfo &st, std::string &json_str) {
    try {
        Json::Value root;
        root["maxObjectCount"] = Json::Value::Int64(st.max_object_count);
        root["maxCapacityMegaBytes"] = Json::Value::Int64(st.max_capacity_mega_bytes);
        for (const auto &each : st.dir_quotas) {
            Json::Value jdir_quota;
            jdir_quota["dirPath"] = Json::Value(each.dir_path).asString();
            jdir_quota["inodeCapacity"] = Json::Value::Int64(each.inode_capacity);
            jdir_quota["storageCapacity"] = Json::Value::Int64(each.storage_capacity);
            // jdir_quota["inode_used"] = each.inode_used;
            // jdir_quota["storage_used"] = each.storage_used;
            root["dirQuotas"].append(jdir_quota);
        }
        Json::FastWriter writer;
        json_str = writer.write(root);
    } catch (...) {
        LOGF(ERROR, "serialize BucketQuotaInfo to json exceptional!");
        return RET_ILLEGAL_OPERATION;
    }
    return RET_OK;
}

int BucketQuotaInfo::deserialize_to_struct(
        Json::Value &root, BucketQuotaInfo &st) {
    try {
        Json::Value &arr = root["dirQuotas"];
        st.max_object_count = root["maxObjectCount"].asInt64();
        st.max_capacity_mega_bytes = root["maxCapacityMegaBytes"].asInt64();
        LOGF(INFO, "origin json str dir_quotas count:%d", arr.size());
        st.dir_quotas.clear();
        for (unsigned int i = 0; i < arr.size(); i++) {
            Json::Value &dir = arr[i];
            BucketDirQuotaInfo dir_st;
            dir_st.dir_path = dir["dirPath"].asString();
            dir_st.inode_capacity = dir["inodeCapacity"].asInt64();
            dir_st.storage_capacity = dir["storageCapacity"].asInt64();
            dir_st.inode_used = dir["inodeUsed"].asInt64();
            dir_st.storage_used = dir["storageUsed"].asInt64();
            st.dir_quotas.push_back(dir_st);
        }
    } catch (...) {
        LOGF(ERROR, "deserialize json to BucketQuotaInfo exceptional!");
        return RET_ILLEGAL_OPERATION;
    }
    return RET_OK;
}

END_CPPSDK_NAMESPACE
